//

// Fragment shader for color adjustment effect
//
// Author: Shirley Carter
//

uniform sampler2DRect texUnit;
uniform float hueVal;
uniform float brightnessVal; 
uniform float contrastVal; 
uniform float saturationVal;  
uniform vec4 textureRect;

uniform int isHueAdjusted;  // 0 = no , 1 = yes 
uniform int isBrightnessAdjusted;  // 0 = no , 1 = yes 
uniform int isContrastAdjusted;  // 0 = no , 1 = yes 
uniform int isSaturationAdjusted;  // 0 = no , 1 = yes 
uniform  vec4 modulateColor;
varying vec2 gUV;
// r,g,b values are from 0 to 1
// h = [0,360], s = [0,1], v = [0,1]
//		if s == 0, then h = -1 (undefined)

vec3 RGBtoHSV( float r, float g, float b )
{
	float minf, maxf, delta;
	vec3 hsv; 

	minf = min(min( r, g), b);
	maxf = max(max( r, g), b);
	hsv.z = maxf; 				// v

	delta = maxf - minf;

	if( abs(maxf) > 0.0 ) {
		hsv.y = delta/maxf;    		// s
		
		if( r == maxf)
			hsv.x = ( g - b ) / delta;		// between yellow & magenta
		else if( g ==maxf )
			hsv.x = 2.0 + ( b - r ) / delta;	// between cyan & yellow
		else
			hsv.x = 4.0 + ( r - g ) / delta;	// between magenta & cyan

		hsv.x = hsv.x*60.0;				// degrees
		if( hsv.x < 0.0 )
			hsv.x = hsv.x+ 360.0;
	}
	else {
		// r = g = b = 0		// s = 0, v is undefined
		hsv.y = 0.0;  // *s = 0;
		hsv.x = -1.0;  //*h = -1;
	}

	return hsv; 
}

vec3 HSVtoRGB( float h, float s, float v )
{
	int i;
	float f, p, q, t;
	vec3 rgb; 

	if( abs(s) > 0.0 ) {
		h  = h/60.0;			// sector 0 to 5
		i = int(floor(h ));
		f = h - floor(h);			// factorial part of h
		p = v * ( 1.0 - s );
		q = v * ( 1.0 - s * f );
		t = v * ( 1.0 - s * ( 1.0 - f ) );

		if ( i == 0 ) {
			rgb.r = v;
			rgb.g = t;
			rgb.b = p;
		}
		else if (i == 1) {
			rgb.r = q;
			rgb.g = v;
			rgb.b = p;
		}
		else if (i == 2 ){
			rgb.r = p;
			rgb.g = v;
			rgb.b = t;
		}
		else if (i == 3){
			rgb.r = p;
			rgb.g = q;
			rgb.b = v;
		}
		else if ( i == 4){
			rgb.r = t;
			rgb.g = p;
			rgb.b = v;
		}
		else {		
			rgb.r = v;
			rgb.g = p;
			rgb.b = q;
		}
	}
	else {
		// achromatic (grey)
		rgb.r = v; 
		rgb.g = v; 
		rgb.b = v; 
	}
	return rgb; 

}

mat3 QuaternionToMatrix(vec4 quat)
{
	vec3 cross = quat.yzx * quat.zxy;
	vec3 square= quat.xyz * quat.xyz;
	vec3 wimag = quat.w * quat.xyz;

	square = square.xyz + square.yzx;

	vec3 diag = 0.5 - square;
	vec3 a = (cross + wimag);
	vec3 b = (cross - wimag);

	return mat3(
	2.0 * vec3(diag.x, b.z, a.y),
	2.0 * vec3(a.z, diag.y, b.x),
	2.0 * vec3(b.y, a.x, diag.z));
}

const vec3 lumCoeff = vec3(0.2125, 0.7154, 0.0721);

void main(void)
{
   if ( gUV.x< textureRect.x || gUV.x> textureRect.x+textureRect.z || gUV.y < textureRect.y || gUV.y > textureRect.y+textureRect.w )
        gl_FragColor = vec4(0.0,0.0,0.0,0.0);
    else {
        vec4 outputColor = texture2DRect(texUnit, gUV);
        vec3 hsv; 
        vec3 intensity; 	 
        
        if (isHueAdjusted ==1 ) {
            vec3 root3 = vec3(0.57735, 0.57735, 0.57735);
            float half_angle = 0.5 * radians(hueVal); // hueVal is radians of 0 tp 360 degree
            vec4 rot_quat = vec4( (root3 * sin(half_angle)), cos(half_angle));
            mat3 rot_Matrix = QuaternionToMatrix(rot_quat);
            
            outputColor.rgb = rot_Matrix * outputColor.rgb;
            
            // here comes the simple way to verify - not as accurate as the above
            //hsv = RGBtoHSV(outputColor.r, outputColor.g, outputColor.b); 
            //hsv.r = fract((hsv.r/360.0)+radians(hueVal))*360.0;
            
            //outputColor.rgb = HSVtoRGB(hsv.r, hsv.g, hsv.b); 
        }
        
        
        if ( isContrastAdjusted == 1 ) {
             outputColor.rgb = (outputColor.rgb - 0.5) *contrastVal + 0.5;
        }
        if ( isBrightnessAdjusted ==1)
            outputColor.rgb = outputColor.rgb +brightnessVal; 
            //pixelColor.rgb += Brightness;

        if ( isSaturationAdjusted==1) {
            intensity = vec3(dot(outputColor.rgb, lumCoeff));
            outputColor.rgb = mix(intensity, outputColor.rgb, saturationVal);  
        }


         // convert grayscale to sepia
        gl_FragColor = modulateColor * outputColor;
    }
}

